USE [NWIC_PanelMgmt]
GO

/****** Object:  StoredProcedure [PanelMgmt].[usp_RebuildAllCDWPatients]    Script Date: 3/23/2017 9:06:41 AM ******/
SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO


CREATE PROCEDURE [PanelMgmt].[usp_RebuildAllCDWPatients]
AS
--==================================================================================
--Requestor             : Non Va Coordinated Care Project, but produces a great
--                        general use all patient table for any application.
--Author                : Jerry Kohler, Jianji Yang - Northwest Innovation Center
--Object/SP Name        : usp_AllCDWPatients
--Server                : DNS.URL        
--Data Base             : NWIC_PanelMgmt
--Schema                : PanelMgmt
--Report                : N/A
--Folder                : N/A
--Report Loc            : N/A
--Job                   : None
--Job Loc               : None
--Note                  : This table contains ALL patients, including
--                        deceased and test.
--Date Created          : 03-10-2016
--
--Last Changed          : 06-06-2016
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Added Next of Kin, and per Microsoft recommendation
--                        changed drop/create table to drop indexes and TRUNCATE.
--                        Also added WITH (TABLOCK) in the insert.
--                        All of this should boost the performance, and not
--                        eliminate stored execution plans on the table.
--
--Last Changed          : 06-03-2016
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Take all patients from SPatient, test or deceased.
--                        Join to PanelMgmt.PMMasterCohortDeceasedIndexNEW
--                        to ensure accuracy.  Add deceased flag and
--                        date of death.
--
--
--Last Changed          : 6-1-2016
--Last Changed By		: Jianji Yang - NWIC
--Reason For Change		: to add PCP, Clinic, HomeSta, and DOB
--Purpose               : To create a table to translate between several types of
--                        patient identification:
--                           - SSN
--                           - SID
--                           - ICN
--                           - Name (last,first)
--                           - CPRSKey (first character of last name, comma, last4)
--Last Changed          : 04-05-2016
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Add test patient data.
--
--Purpose               : Because SPatient does not have an index on PatientSSN,
--                        we create a table to translate between several types of
--                        patient identification:
--                           - SSN
--                           - SID
--                           - ICN
--                           - Name (last,first)
--                           - CPRSKey (first character of last name, last4)
--
--Last Changed          : 04-05-2016
--Last Changed By		: Jianji Yang - NWIC
--Reason For Change		: Add test patient data, deceased patients, and patient's
--                      : home station (based on PCP's Sta3n).
--
--Last Changed          : 03-17-2017
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Add PatientIEN so that function App.udf_AllPatientsSIDs
--                      : can now hit AllCDWPatients instead of SPatient.  This
--                      : table is much smaller, and will be much faster.
--
--Last Changed          : 03-20-2017
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Change procedure to place data in a temporary table THEN
--                      : truncate and insert into the production table.  This
--                      : minimize the amount of time queries would return no data.
--==================================================================================

BEGIN

    -- Create temporary tables

    -- Rank PCP Relationship Start Date latest to oldest.
    -- We want the most recently assigned Primary Care Provider.
    -- The Provider's station will become the patient's
    -- home station (patient parent site)
    select ROW_NUMBER() over(Partition by PatientSID ORDER BY RelationshipStartDate Desc) As RowNum
          ,PatientSID
          ,[PrimaryProviderSID]
          ,PrimaryProvider as [PrimaryProviderName]
          ,[TeamSID]
          ,Team as [PCPTeam]
          ,InstitutionCode as [Sta6a]
          ,DivisionName as [Clinic]
    into #PCP
    from [App].[PCMMCurrentPatientProviderRelationship];

    create unique clustered index #PCPidx on #PCP(PatientSID, RowNum) with (SORT_IN_TEMPDB = ON);


    -- Next of Kin for REFDOC
    IF (SELECT OBJECT_ID('#NOK')) IS NOT NULL
    DROP TABLE #NOK

    Select a.PatientSID, a.SPatientAddressSid
    Into #NOK
    From CDWWork.SPatient.SPatientAddress a
    Where PatientSID > 0
      and a.OrdinalNumber = (select min(x.OrdinalNumber)
                             from cdwwork.SPatient.SPatientAddress x
                             where x.PatientSID = a.PatientSID)
      and a.AddressType like '%Next Of Kin%';

    Create unique clustered index #spa ON #NOK (PatientSID, SPatientAddressSID) with (SORT_IN_TEMPDB = ON);



    -- Get patient data and put in a temporary table
    IF (SELECT OBJECT_ID('#PTS')) IS NOT NULL
    DROP TABLE #PTS

    select a.[Sta3n]
          ,a.[PatientIEN]
          ,a.[PatientSSN]
          ,a.[PatientSID]
          ,a.[PatientICN]
          ,a.[PatientName]
          ,a.DateOfBirth
          ,RIGHT(a.[PatientSSN], 4) as LastFour
          ,LEFT(a.[PatientLastName], 1) + RIGHT(a.[PatientSSN], 4) AS [CPRSKey]
          ,ISNULL(c.[PrimaryProviderSID] , 0) AS PrimaryProviderSID
          ,ISNULL(c.[PrimaryProviderName], 'PCP Not Assigned') AS PrimaryProviderName
          ,c.TeamSID AS [PCPTeamSID]
          ,c.[PCPTeam]
          ,c.[Sta6a]
          ,c.[Clinic]
          ,NULL as HomeSta3n
          ,[Deceased] = CASE
                            When b.DateOfDeath is NOT NULL then 1
                            Else 0
                        END
		  ,b.[DateOfDeath]
          ,[TestPatientFlag] = CASE
                                   When a.[TestPatientFlag] IS NULL Then 0
                                   Else 1
                               END
          ,d.SpatientAddressSID AS [NOKAddressSID]    -- Next of Kin is in CDWWork.SPatient.SPatientAddress
    into #PTS
    from CDWWork.SPatient.SPatient a
	LEFT JOIN PanelMgmt.PMMasterCohortDeceasedIndexNEW b on b.PatientSID = a.PatientSID
	LEFT JOIN #PCP c on a.PatientSID = c.PatientSID AND c.RowNum = 1
    LEFT JOIN #NOK d on a.PatientSID = d.PatientSID
    where a.[PatientSSN] Not In ('*Missing*', '*Unknown at this time*')

    Create nonclustered index #PTS1 on #PTS (PatientICN)
    with (SORT_IN_TEMPDB = ON)
    
    Create nonclustered index #PTS2 on #PTS (PatientSID)
    with (SORT_IN_TEMPDB = ON)
    
    Create nonclustered index #PTS3 on #PTS (PrimaryProviderSID)
    with (SORT_IN_TEMPDB = ON)


    -- Update PCP and Home Station columns

    Update #PTS
	set [PrimaryProviderSID] = 0,
	[PrimaryProviderName] ='PCP Not Assigned'
	where [PrimaryProviderSID] < 0


	select distinct sta3n, patienticn 
    into #homestation
	from #PTS
	where [PrimaryProviderSID] > 0
    
    Create clustered index #HomeStationICN on #homestation (PatientICN)
        
    update  #PTS
	set HomeSta3n =b.Sta3n
	from #PTS a
	join #homeStation b
	on a.patienticn = b.patienticn


    -- Drop all indexes except the primary clustered index (create statement
    -- below in case it gets dropped to avoid the overhead of maintaining
    -- them during the truncate and insert of the table.  This method was
    -- enacted based on CDW operational recommendations by Microsoft.

    IF (SELECT OBJECT_ID('App.AllCDWPatients')) IS NOT NULL
        BEGIN
            DROP index AllCDWPatients_PatientICN         on App.AllCDWPatients;
            DROP index AllCDWPatients_PatientSID         on App.AllCDWPatients;
            DROP index AllCDWPatients_PatientName        on App.AllCDWPatients;
            DROP index AllCDWPatients_CPRSKey            on App.AllCDWPatients;
            DROP index AllCDWPatients_PrimaryProviderSID on App.AllCDWPatients;

            TRUNCATE TABLE App.AllCDWPatients;
        END

    
    -- Insert patient records into the table

    Insert into App.AllCDWPatients WITH (TABLOCK)
        (
        [Sta3n],
        [PatientIEN],
       	[PatientSSN],
       	[PatientSID],
       	[PatientICN],
        [PatientName],
		[DateOfBirth],
		[LastFour],
        [CPRSKey],
		[PrimaryProviderSID],
		[PrimaryProviderName],
		[PCPTeamSID],
		[PCPTeam],
		[Sta6a],
		[Clinic],
		[HomeSta3n],
		[Deceased],
		[DateOfDeath],
        [TestPatientFlag],
        [NOKAddressSID]
        )
    (
    select [Sta3n],
           [PatientIEN],
       	   [PatientSSN],
       	   [PatientSID],
       	   [PatientICN],
           [PatientName],
		   [DateOfBirth],
		   [LastFour],
           [CPRSKey],
		   [PrimaryProviderSID],
		   [PrimaryProviderName],
		   [PCPTeamSID],
		   [PCPTeam],
		   [Sta6a],
		   [Clinic],
		   [HomeSta3n],
		   [Deceased],
		   [DateOfDeath],
           [TestPatientFlag],
           [NOKAddressSID]
      from #PTS
    )


    -- Recreate indexes

    -- In case this one gets dropped somehow.  This one is
    -- critical to REFDOC (aka CITCoM, aka NVCC)

    --Create Clustered Index AllCDWPatients_Sta3n_PatientSSN
    --On App.AllCDWPatients (Sta3n, PatientSSN)
    --With (SORT_IN_TEMPDB = ON, DATA_COMPRESSION = PAGE)


    Create nonclustered index AllCDWPatients_PatientICN on App.AllCDWPatients (PatientICN)
    include (PatientSID, PatientSSN, Sta3n, Deceased, DateOfDeath)
    with (SORT_IN_TEMPDB = ON, DATA_COMPRESSION = PAGE)
    
    Create nonclustered index AllCDWPatients_PatientSID on App.AllCDWPatients (PatientSID)
    include (PatientICN, Deceased, DateOfDeath)
    with (SORT_IN_TEMPDB = ON, DATA_COMPRESSION = PAGE)
    
    Create nonclustered index AllCDWPatients_PatientName on App.AllCDWPatients (PatientName)
    include (PatientICN, PatientSID, PatientSSN, Sta3n, Deceased, DateOfDeath)
    with (SORT_IN_TEMPDB = ON, DATA_COMPRESSION = PAGE)
    
    Create nonclustered index AllCDWPatients_CPRSKey on App.AllCDWPatients (CPRSKey)
    include (PatientICN, PatientSID, PatientSSN, Sta3n, Deceased, DateOfDeath)
    with (SORT_IN_TEMPDB = ON, DATA_COMPRESSION = PAGE)
    
    Create nonclustered index AllCDWPatients_PrimaryProviderSID on App.AllCDWPatients (PrimaryProviderSID)
    include (PrimaryProviderName, PatientICN, PatientSID)
    with (SORT_IN_TEMPDB = ON, DATA_COMPRESSION = PAGE)
    

    -- Clean up

    IF (SELECT OBJECT_ID('#PCP')) IS NOT NULL
		DROP TABLE #PCP;

    IF (SELECT OBJECT_ID('#NOK')) IS NOT NULL
        DROP TABLE #NOK

    IF (SELECT OBJECT_ID('#homestation')) IS NOT NULL
        DROP TABLE #homestation

    IF (SELECT OBJECT_ID('#PTS')) IS NOT NULL
    DROP TABLE #PTS

END

GO
